/******************************************************************************
	TalkBot_Slave7.c      v1.0    29th July 2009 - Roman Black
	Open-source-firmware for TalkBotBrain product from BlackRobotics.com
	This firmware is open-source for use by legitimate owners
	who purchased a TalkBotBrain. All rights to the TalkBot hardware
	circuitry design remain property of BlackRobotics.

	This is TalkBot_Slave7 - It is a simple sound player,
	it does not drive servos. When BUT1 is pressed, or
	an external button on PORTB.5 goes low, Slave7 plays
	the next sound in sequence. (Max sounds 255)
	It does not use RX or TX serial commands. It still does serial
	download of sound library from PC at 19200 baud.

	Slave7 uses a special function; get_number_sounds()
	to read the external eeprom header and find out how many
	sounds are stored in the sound library.

	TalkBot1 uses a PIC 16F628 and 2x 24LC512 i2c eeproms (1Megabit).

	Slave 7; The external button is on;
	 PORTB 5 (button is active low. Low = play the next sound)

	For more TalkBot applications and FREE Sound Library files;
	 www.TalkBotBrain.com

// NOTE! Set TABS to 4
******************************************************************************/
/*
_BODEN_OFF   // these configs are set in MikroC compiler options.
_PWRTE_ON
_WDT_ON
_LVP_OFF
_MCLRE_OFF
_HS_OSC
*/
// vars in general operation, main() and int() etc;

#define PIN_LED1 PORTB.F3	// hi=ON, shared with BTc sound out
#define PIN_LED2 PORTB.F2	// lo=ON, shared with USART TX out, disabled if USART on

#define PIN_BUT1 PORTB.F1	// lo=pressed, shared with USART RX in
#define PIN_BUT2 PORTA.F5	// lo=pressed, general use button

unsigned char t0_toggle;	// for period timing with TMR0
unsigned char t0_count;		// period timing, debouncing

//unsigned char RXbyte1;		// the 2-byte command received by RX serial
//unsigned char RXbyte2;		// (not used in this version Slave)

unsigned char whichsound;	// which sound to play when button2 pressed
unsigned int temp16;		// used for any 16bit math

// used for Slave6 only;
unsigned char numsounds;	// how many sounds are in Sound Library

//---------------------------------------------------------
// (Servo8 system is not used in this version Slave)

//---------------------------------------------------------
// used in RI2C.C for my i2c eeprom functions

unsigned char ri2c_i absolute 0x20;	// MUST be in register bank0
unsigned char data;

unsigned char ri2c_add_hi;		// 16bit address within external i2c eeprom
unsigned char ri2c_add_lo;		//
unsigned char ri2c_add_chip;	//

#define	RBTC_PIN_SDA	PORTA.F4	// SDA pin, i2c eeprom
#define	RBTC_TRIS_SDA	TRISA.F4	// SDA pin, i2c eeprom
#define	RBTC_PIN_SCL	PORTB.F0	// SCL pin, i2c eeprom
#define	RBTC_PIN_BTC	PORTB.F3	// sound output pin

#define NO_ACK	0				// for i2c to control ack bit
#define ACK		1

// functions in RI2C.c

void ri2c_start_bit(void);
void ri2c_stop_bit(void);
void ri2c_send_byte(void);
void ri2c_receive_byte(unsigned char);

//---------------------------------------------------------
// vars in RBTC.c for BTc 1bit sound playback

#define	RBTC_SDA	PORTA,4			// the same 4 pin defines, syntax ASM only
#define	RBTC_TSDA	TRISA,4			//
#define	RBTC_SCL	PORTB,0			//
#define	RBTC_BTC	PORTB,3			//

// functions in RBTC.c
void play_btc_lib_sound(unsigned char thesound);

//---------------------------------------------------------
// vars in RSTREAM.c needed for serial data stream to eeprom

unsigned char timeout;			// timeout when stream is finished
unsigned char pos_in;			// pointer to position in input buffer
unsigned char input_full;		// flag to show we have 16 more input bytes

unsigned char in_buffer[32];	// buffer to hold incoming serial data stream
unsigned char out_buffer[32] absolute 0xB0;	// data being sent out to I2C eeprom

// functions in RSTREAM.c
void poll_serial_in(void);
void receive_stream(void);
void send_stream(void);
void rdelay_ms(unsigned char);	// used for mS delays anywhere

//-----------------------------------------------------------------------------


//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void  interrupt(void)
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
{
	//----------------------------------------------------
	// (interrupt is not used in this version Slave)
}
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=============================================================================
#include "RI2C.c"		// my i2c eeprom functions
//=============================================================================
#include "RSTREAM.c"	// functions to do serial port data streaming
//=============================================================================
#include "RBTC.c"		// functions to do BTc 1bit sound playback
//=============================================================================

//=============================================================================
//  GET_NUMBER_SOUNDS
//=============================================================================
void get_number_sounds(void)
{
	//-------------------------------------------------
	// Used in TalkBot Slave6 only!
	//
	// This function reads the Sound Library header data
	// stored in external eeprom and finds out how many
	// sounds are in the sound library.
	// The number is then stored in var numsounds
	//-------------------------------------------------
	unsigned int the_address;

	numsounds = 0;

	// setup eeprom addressing ready to read the header
	ri2c_add_hi = 0;
	ri2c_add_lo = 0;    // to read the address of first sound

	// loop and count each sound that has a valid address
	while(numsounds < 255)
	{
		asm clrwdt;
		//---------------------------
		// read the 2 byte address from external eeprom
		ri2c_start_bit();

		data = 0b10100000;	// control byte 1010 AAA WRITE
		ri2c_send_byte();
		data = ri2c_add_hi;   // address of first byte to read
		ri2c_send_byte();
		data = (ri2c_add_lo + 2);
		ri2c_send_byte();

		// tell the eeprom we want to read from it now
		ri2c_start_bit();
		data = 0b10100001;	// control byte 1010 AAA READ
		ri2c_send_byte();

		// read the 2 bytes into the 16bit address
		ri2c_receive_byte(ACK);			// get byte1
		the_address = (data * 256);
		ri2c_receive_byte(NO_ACK);		// get byte2, NO ack on last read!!
		the_address += data;

		// shut eeprom down
		ri2c_stop_bit();

		//---------------------------
		// if it is a valid sound, its address will be >0 and <0xFFFF
		if((the_address > 0) && (the_address < 0xFFFF))
		{
			numsounds++;    // found 1 more sound!

			ri2c_add_lo += 4;   // move up 4 ready to read next address
			if(ri2c_add_lo == 0) ri2c_add_hi++;
		}
		else		// else we failed to find a sound!
		{
			return;		// so it is done
		}
	}
}
//-----------------------------------------------------------------------------


//=============================================================================
//  MAIN
//=============================================================================
void main()
{
	//-------------------------------------------------
	// setup the PIC port registers etc

	CMCON = 0x07;			// set all pins to digital, comparators off

	PORTA =  0b00000000;	// RA4 must be kept low
	TRISA  = 0b00110000;  	// RA0-3 are 4 unused outputs
							// RA4 is SDA to eeprom, high imped
							// RA5 is but2 in

	PORTB =  0b00000101;	// make RB2 TX out high not-active
	TRISB  = 0b00100010;   	// RB0 is SCL out, RB1 is RX in
							// RB2 is TX out, RB3 is sound out
	                        // RB5 is the external input button
							// RB4,6,7 unused outputs
	//-------------------------------------------------
 	// TIMER setups (PIC is 20MHz xtal)

	// TMR0 is used for general timing; mS delays, debouncing
	OPTION_REG = 0b00000111;	// TMR0 ON, at 256:1, PORTB pullups ON

	// TMR1 is used in interrupt for Servo8 system
	T1CON = 0b00000001;		// TMR1 ON, at 1:1 (5MHz)

	// TMR2 is used in BTc sound playback to control bitrate
	T2CON = 0b00000100;		// TMR2 ON, at 1:1 (5MHz)

   	//-------------------------------------------------
	// now TMR0 is operating, make a small delay to let PIC pins settle
	rdelay_ms(100);

	// and make a LED flash (and distinctive sound beep) to show
	// everything powered up ok. we can re-use vars t0_count
	// t0_toggle and whichsound here as they are not needed yet.

	whichsound = 3;			// 3 beeps
	while(whichsound)
	{
		t0_count = 25;  	// of 25 LED pulses
		t0_toggle = 6;		// starting at 6mS period
		while(t0_count)
		{
			// pulse the LED (and sound! is on same pin)
			rdelay_ms(t0_toggle);   // LED off period
			PIN_LED1 = 1;
			rdelay_ms(1);			// LED on period
			PIN_LED1 = 0;

			if(t0_toggle >1) t0_toggle--;
			t0_count--;
		}
		whichsound--;
	}

   	//-------------------------------------------------
	// (serial RX/TX commands not used in this version Slave)

   	//-------------------------------------------------
	// (TMR1 interrupt is not used in this version Slave)
	INTCON = 0;		// all ints OFF

   	//-------------------------------------------------
	// setup any variables
	
	whichsound = 0;		// sound0 is first to play if button2 pressed
	t0_count = 0;		// clear button2 debounce counter

   	//-------------------------------------------------
	// Slave7 only!
	// find out how many sounds are in sound library
	
	rdelay_ms(250);			// small delay to make sure power is stable
	get_number_sounds();

   	//-------------------------------------------------
	// TalkBot main run loop here
	while(1)
	{

		//-------------------------------------------------
		// if BUT1 is pressed or external button on PORTB.5
		// is pressed then play the next sound
		if(!PIN_BUT1 || !PORTB.F5)
		{
			play_btc_lib_sound(whichsound);
			whichsound++;
			if(whichsound >= numsounds) whichsound = 0;
		}

		//-------------------------------------------------
		// if BUT2 is held down for >3 seconds,
		// go into receive serial stream mode
		if(!PIN_BUT2)		// if but2 pressed
		{
			t0_count++;
			if(t0_count >= 166)		// if but2 pressed >3 seconds
			{
				t0_count = (152-20-1);	// make it flash LED straight away

				// go into serial receive mode here.
				// first turn USART on at 19200 baud!
				RCSTA = 0;
				TXSTA = 0b00100100;		// high BRGH mode
				SPBRG = 64;				// at 20Mhz gives 19231 baud
				RCSTA = 0b10010000;		// SPEN bit7 =1 enables serial port

				// flash LED1 and wait for first serial byte to arrive
				while(1)
				{
					asm clrwdt;
					// flash LED1 once per second; 152Hz/152
					if(TMR0.F7 != t0_toggle.F0)
                	{
						t0_toggle++;
						t0_count++;
						if(t0_count == (152-20)) PIN_LED1 = 1; // LED1 ON
						if(t0_count >= 152)
						{
							t0_count = 0;
							PIN_LED1 = 0;	// LED1 OFF
						}
					}
					// check for first serial byte received
					if(PIR1.RCIF == 1)
					{
						receive_stream();
						t0_count = 0;
						whichsound = 0;		// ready to play first sound
					    break;
					}
				}

				// receive is done, turn USART off again...
				RCSTA = 0;
			}
		}
		else    // else BUT2 is released
		{
			// if BUT2 was quick pressed, play a sound!
			if(t0_count > 1)	// 2 gives small BUT2 debounce
			{
				play_btc_lib_sound(whichsound);		// play a sound
				whichsound++;		// select next sound (full 0-255 range)

				// Slave6 only, auto reset sounds after last sound
				if(whichsound >= numsounds) whichsound=0;
			}
			t0_count = 0;	// reset BUT2 debounce count
		}
	   	//-------------------------------------------------
		// To reduce power consumption, put the PIC in sleep
		// mode. The watchdog will wake it up after 18mS.

		asm sleep;

	   	//-------------------------------------------------
	}
}
//-----------------------------------------------------------------------------






